#!/bin/sh
# -----------------------------------------------------------------------------
# Control FAC Node
#
# Copyright: © 2015-2020 Claris International Inc. All rights reserved.
# Environment Variable Prerequisites
#
#   ARGS     start, stop.
#
# -----------------------------------------------------------------------------

# FileMaker FAC_NODE Settings
export NODE_ENV=production
export FAC_SESSION_SECRET=`openssl rand -base64 32`
FAC_NODE_BASE=$(dirname "$0")
	
case "$(uname -s)" in
	Darwin)
		SERVER_INSTALL_DIR="/Library/FileMaker Server"
		PLATFORM="macOS"
		FORCEKILL_CMD="kill -USR1"
		KILL_CMD="kill" 
		;;
	Linux)
		SERVER_INSTALL_DIR="/opt/FileMaker/FileMaker Server"
		PLATFORM="Linux"
		FORCEKILL_CMD="pkill -P"
		KILL_CMD="pkill -P" 
		;;
esac

FormatDate()
{
	# Gets current time, formatted
	date "+%Y-%m-%d %H:%M:%S %z"
}

if [ -d "$SERVER_INSTALL_DIR/Admin/SAC" ]; then
		FAC_DIR="$SERVER_INSTALL_DIR/Admin/SAC"
else
		FAC_DIR="$SERVER_INSTALL_DIR/Admin/FAC"
fi

FAC_NODEJS="$FAC_DIR/facstart.sh"
FAC_LOG="$FAC_DIR/logs/fac.log"
FAC_NODE_PID=""

if [ "$PLATFORM" = "macOS" ] ; then
		# Verify that facstart.sh is a hardlink to the appropriate fac-x64 vs fac-arm64 binary
		FAC_PKG="$FAC_DIR/fac-x64"
		if [ "$(arch)" == "arm64" ]; then
				FAC_PKG="$FAC_DIR/fac-arm64"
		fi
		if [ ! -f "$FAC_PKG" ]; then
				echo "$(FormatDate)	FATAL - $FAC_PKG not found!" >> "$FAC_LOG"
				exit 1
		fi

		FAC_UPDATEREQUIRED=1
		if [ -f "$FAC_NODEJS" ]; then
				echo "$(FormatDate)	facstart.sh is present - verifying contents" >> "$FAC_LOG"
				if [ "$FAC_PKG" -ef "$FAC_NODEJS" ]; then
						echo "$(FormatDate)	facstart.sh matches $FAC_PKG" >> "$FAC_LOG"
						FAC_UPDATEREQUIRED=0
				fi
		fi

		if [ "$FAC_UPDATEREQUIRED" = "1" ]; then
				echo "$(FormatDate)	Updating facstart.sh to hard-link to $FAC_PKG" >> "$FAC_LOG"
				rm -rf "$FAC_NODEJS"
				ln "$FAC_PKG" "$FAC_NODEJS"
		fi
fi

if [ ! -f "$FAC_NODEJS" ]; then
	echo "$(FormatDate) $FAC_NODEJS not found!" >> "$FAC_LOG"
	exit 1
fi

if [ "$1" = "start" ] ; then

	if [ "$PLATFORM" = "Linux" ] ; then
		echo "$(FormatDate)	Starting FAC node" >> "$FAC_LOG"
	else
		echo "$(FormatDate)	Starting FAC node"
	fi
	shift
	
	if [ "$1" = "-p" ]; then
		shift
		FAC_NODE_BASE=$1
	fi

	if [ "$PLATFORM" = "Linux" ] ; then
		echo "$(FormatDate)	Execute $FAC_NODEJS" >> "$FAC_LOG"
	else
		echo "$(FormatDate)	Execute $FAC_NODEJS"
	fi
#
# Use exec to launch the program to replace this shell process so we do not have a shell in between.
#
	exec "$FAC_NODEJS" 

elif [ "$1" = "stop" ] ; then

	if [ "$PLATFORM" = "Linux" ] ; then
		echo "$(FormatDate)	Stopping FAC node" >> "$FAC_LOG"
	else
		echo "$(FormatDate)	Stopping FAC node"
	fi

	shift
	FORCE=0
	if [ "$1" = "-force" ]; then
		shift
		FORCE=1
	fi

	if [ "$1" = "-z" ]; then
		shift
		FAC_NODE_PID=$1
	fi
	# we don't pass pid, can we assume we are kill all node

# check if FAC_NODE_PID exists and is node. to make sure we don't kill random process
# we can specify to send specific custom signal like SIGFMND (signal FM node) function
# and we can call kill $FAC_NODE_PID

# echo "$FAC_NODE_PID"
	if [ "$FAC_NODE_PID" = "" ]; then
		echo "$(FormatDate)	PID must be specified to stop!"
		# do nothing, because we assume this its already killed
	else

		FAC_PID_EXIST=$(ps "$FAC_NODE_PID")

		if [ "$FAC_PID_EXIST" != "" ]; then

		 # force kill
			if [ $FORCE -eq 1 ]; then
				if [ "$PLATFORM" = "Linux" ] ; then
					echo "$(FormatDate)	Killing with force: $FAC_NODE_PID" >> "$FAC_LOG"
				else
					echo "$(FormatDate)	Killing with force: $FAC_NODE_PID"
				fi
				$FORCEKILL_CMD "$FAC_NODE_PID"

			else
				if [ "$PLATFORM" = "Linux" ] ; then
					echo "$(FormatDate)	Killing: $FAC_NODE_PID" >> "$FAC_LOG"
				else
					echo "$(FormatDate)	Killing: $FAC_NODE_PID"
				fi
				$KILL_CMD "$FAC_NODE_PID"
			fi
		else
			echo "$(FormatDate)	PID $FAC_NODE_PID is not running or is not 'FAC node'"
		fi

	fi

else

	echo "Usage: $0 ( commands ... )"
	echo "commands:"
	echo "  start               Start FAC node in a separate window"
	echo "  stop -z PID         Stop FAC node with process id = PID"
	echo "  stop -force -z PID  Stop FAC node with process id = PID forcefully"
	exit 1

fi
exit 0
